import Foundation
import HealthKit

/// Comprehensive health data snapshot containing all fetched metrics
struct HealthDataSnapshot {
    // MARK: - Health Metrics
    let restingHeartRate: Double?
    let heartRateVariabilitySDNN: Double?
    let vo2Max: Double?
    let bodyMass: Double?
    let bodyMassIndex: Double?
    let stepCount: Double?
    let sleepAnalysis: SleepData?
    let biologicalSex: HKBiologicalSex?
    let dateOfBirth: Date?
    
    // MARK: - Metadata
    let timestamp: Date
    let dataSource: String
    let availableDataTypes: Set<String>
    let missingDataTypes: Set<String>
    
    // MARK: - Computed Properties
    var isComplete: Bool {
        return missingDataTypes.isEmpty
    }
    
    var completenessPercentage: Double {
        let totalTypes = availableDataTypes.count + missingDataTypes.count
        return totalTypes > 0 ? Double(availableDataTypes.count) / Double(totalTypes) : 0.0
    }
    
    var hasCriticalData: Bool {
        // Essential data for basic life expectancy calculations
        return restingHeartRate != nil && 
               biologicalSex != nil && 
               dateOfBirth != nil
    }
    
    var age: Int? {
        guard let dateOfBirth = dateOfBirth else { return nil }
        return Calendar.current.dateComponents([.year], from: dateOfBirth, to: Date()).year
    }
    
    // MARK: - Initialization
    init(
        restingHeartRate: Double? = nil,
        heartRateVariabilitySDNN: Double? = nil,
        vo2Max: Double? = nil,
        bodyMass: Double? = nil,
        bodyMassIndex: Double? = nil,
        stepCount: Double? = nil,
        sleepAnalysis: SleepData? = nil,
        biologicalSex: HKBiologicalSex? = nil,
        dateOfBirth: Date? = nil,
        dataSource: String = "HealthKit"
    ) {
        self.restingHeartRate = restingHeartRate
        self.heartRateVariabilitySDNN = heartRateVariabilitySDNN
        self.vo2Max = vo2Max
        self.bodyMass = bodyMass
        self.bodyMassIndex = bodyMassIndex
        self.stepCount = stepCount
        self.sleepAnalysis = sleepAnalysis
        self.biologicalSex = biologicalSex
        self.dateOfBirth = dateOfBirth
        self.timestamp = Date()
        self.dataSource = dataSource
        
        // Determine available and missing data types
        var available = Set<String>()
        var missing = Set<String>()
        
        let dataChecks: [(String, Bool)] = [
            ("restingHeartRate", restingHeartRate != nil),
            ("heartRateVariabilitySDNN", heartRateVariabilitySDNN != nil),
            ("vo2Max", vo2Max != nil),
            ("bodyMass", bodyMass != nil),
            ("bodyMassIndex", bodyMassIndex != nil),
            ("stepCount", stepCount != nil),
            ("sleepAnalysis", sleepAnalysis != nil),
            ("biologicalSex", biologicalSex != nil),
            ("dateOfBirth", dateOfBirth != nil)
        ]
        
        for (type, isAvailable) in dataChecks {
            if isAvailable {
                available.insert(type)
            } else {
                missing.insert(type)
            }
        }
        
        self.availableDataTypes = available
        self.missingDataTypes = missing
    }
    
    // MARK: - Fallback Calculations
    func estimatedBMI(height: Double?) -> Double? {
        guard let mass = bodyMass, let height = height, height > 0 else { return nil }
        return mass / (height * height)
    }
    
    func healthScore() -> Double {
        var score: Double = 0.0
        var factors: Int = 0
        
        // Resting Heart Rate (lower is better, optimal range: 60-100 bpm)
        if let rhr = restingHeartRate {
            let normalizedRHR = max(0, min(1, (100 - rhr) / 40)) // Normalize to 0-1
            score += normalizedRHR
            factors += 1
        }
        
        // HRV (higher is typically better)
        if let hrv = heartRateVariabilitySDNN {
            let normalizedHRV = min(1, hrv / 50) // Normalize assuming 50ms as good
            score += normalizedHRV
            factors += 1
        }
        
        // VO2 Max (higher is better)
        if let vo2 = vo2Max {
            let normalizedVO2 = min(1, vo2 / 50) // Normalize assuming 50 as excellent
            score += normalizedVO2
            factors += 1
        }
        
        // BMI (optimal range around 18.5-24.9)
        if let bmi = bodyMassIndex {
            let optimalBMI = 21.7 // Middle of healthy range
            let deviation = abs(bmi - optimalBMI) / optimalBMI
            let normalizedBMI = max(0, 1 - deviation)
            score += normalizedBMI
            factors += 1
        }
        
        // Sleep quality (if available)
        if let sleep = sleepAnalysis {
            score += sleep.qualityScore
            factors += 1
        }
        
        return factors > 0 ? score / Double(factors) : 0.0
    }
}

/// Sleep analysis data structure
struct SleepData {
    let totalSleepHours: Double
    let deepSleepPercentage: Double?
    let remSleepPercentage: Double?
    let sleepEfficiency: Double?
    
    var qualityScore: Double {
        var score: Double = 0.0
        var factors: Int = 0
        
        // Sleep duration (optimal: 7-9 hours)
        let durationScore = max(0, min(1, 1 - abs(totalSleepHours - 8) / 4))
        score += durationScore
        factors += 1
        
        // Sleep efficiency (if available)
        if let efficiency = sleepEfficiency {
            score += min(1, efficiency / 0.9) // 90% efficiency as optimal
            factors += 1
        }
        
        return factors > 0 ? score / Double(factors) : 0.5 // Default middle score
    }
}

// MARK: - Extensions for Better Debugging
extension HealthDataSnapshot: CustomStringConvertible {
    var description: String {
        var desc = "HealthDataSnapshot (timestamp: \(timestamp))\n"
        desc += "Completeness: \(Int(completenessPercentage * 100))% (\(availableDataTypes.count)/\(availableDataTypes.count + missingDataTypes.count) data types)\n"
        
        if let age = age {
            desc += "Age: \(age) years\n"
        }
        
        if let rhr = restingHeartRate {
            desc += "Resting HR: \(Int(rhr)) bpm\n"
        }
        
        if let hrv = heartRateVariabilitySDNN {
            desc += "HRV SDNN: \(Int(hrv)) ms\n"
        }
        
        if let vo2 = vo2Max {
            desc += "VO2 Max: \(String(format: "%.1f", vo2)) ml/kg/min\n"
        }
        
        if let bmi = bodyMassIndex {
            desc += "BMI: \(String(format: "%.1f", bmi))\n"
        }
        
        if let steps = stepCount {
            desc += "Daily Steps: \(Int(steps))\n"
        }
        
        if !missingDataTypes.isEmpty {
            desc += "Missing: \(missingDataTypes.sorted().joined(separator: ", "))\n"
        }
        
        return desc
    }
}